package com.flowkraft

import grails.config.Config
import grails.core.support.GrailsConfigurationAware
import groovy.json.JsonSlurper
import groovy.util.logging.Slf4j
import groovy.xml.XmlSlurper

/**
 * Service for communicating with the ReportBurster backend API.
 * 
 * This service proxies requests to the ReportBurster Spring Boot backend.
 * 
 * AUTO-DISCOVERY:
 * - Backend URL: /app/config/settings.xml → <backendurl>http://localhost:9090/api</backendurl>
 * - API Key: /app/config/api-key.txt (UUID generated by Spring Boot on first startup)
 * 
 * This ensures the Grails app always uses the correct port and API key, even when the
 * server dynamically chooses a port (9090, 9091, etc.) based on availability.
 * 
 * SECURITY:
 * - All requests include the X-API-Key header for authentication
 * - The API key is auto-discovered from api-key.txt (no manual configuration needed)
 * 
 * API Endpoints:
 * - GET /jobman/reporting/fetch-data - Fetch report data with optional parameters
 * 
 * Response Format (ReportDataResult):
 * {
 *   "reportData": [...],          // Array of row objects
 *   "reportColumnNames": [...],   // Array of column names
 *   "executionTimeMillis": 123,   // Query execution time
 *   "isPreview": true,            // Whether this is a preview
 *   "totalRows": 100              // Total row count
 * }
 */
@Slf4j
class ReportBursterService implements GrailsConfigurationAware {

    String backendBaseUrl
    String defaultConfigPath
    String apiKey
    
    // Path to settings.xml relative to the Grails app root
    static final String SETTINGS_XML_PATH = '/app/config/settings.xml'
    
    // Path to API key file relative to the Grails app root
    static final String API_KEY_PATH = '/app/config/api-key.txt'

    @Override
    void setConfiguration(Config config) {
        // First try to read from settings.xml (auto-discovery)
        backendBaseUrl = discoverBackendUrl()
        
        // Fallback to application.yml config if settings.xml not available
        if (!backendBaseUrl) {
            backendBaseUrl = config.getProperty('reportburster.backend.baseUrl', String, 'http://localhost:9090')
            log.warn("Could not read settings.xml, using fallback URL: ${backendBaseUrl}")
        }
        
        // Auto-discover API key from api-key.txt
        apiKey = discoverApiKey()
        
        defaultConfigPath = config.getProperty('reportburster.backend.defaultConfigPath', String, '/app/config/settings.xml')
    }
    
    /**
     * Auto-discover backend URL from ReportBurster's settings.xml.
     * 
     * The settings.xml is maintained by the ReportBurster startup scripts,
     * which update <backendurl> with the actual port the server started on.
     * 
     * @return The backend base URL (e.g., "http://localhost:9090") or null if unavailable
     */
    private String discoverBackendUrl() {
        try {
            // Resolve path relative to grails-app directory
            def settingsFile = new File(System.getProperty('user.dir'), SETTINGS_XML_PATH)
            
            if (!settingsFile.exists()) {
                // Try alternate path from application root
                settingsFile = new File('/app/config/settings.xml')
            }
            
            if (!settingsFile.exists()) {
                log.warn("settings.xml not found at expected paths")
                return null
            }
            
            def xml = new XmlSlurper().parse(settingsFile)
            def backendUrl = xml.settings.backendurl?.text()?.trim()
            
            if (backendUrl) {
                // The backendurl in settings.xml includes "/api" suffix, 
                // but we need the base URL without it for our API calls
                // Example: "http://localhost:9090/api" → "http://localhost:9090"
                if (backendUrl.endsWith('/api')) {
                    backendUrl = backendUrl[0..-5]  // Remove "/api"
                }
                log.info("Auto-discovered backend URL from settings.xml: ${backendUrl}")
                return backendUrl
            }
            
            log.warn("No backendurl found in settings.xml")
            return null
            
        } catch (Exception e) {
            log.warn("Error reading settings.xml: ${e.message}")
            return null
        }
    }
    
    /**
     * Auto-discover API key from ReportBurster's api-key.txt file.
     * 
     * The api-key.txt is created by the Spring Boot backend on first startup.
     * It contains a UUID that must be sent in the X-API-Key header.
     * 
     * @return The API key or null if unavailable
     */
    private String discoverApiKey() {
        try {
            // Resolve path relative to grails-app directory
            def apiKeyFile = new File(System.getProperty('user.dir'), API_KEY_PATH)
            
            if (!apiKeyFile.exists()) {
                // Try alternate path from application root
                apiKeyFile = new File('/app/config/api-key.txt')
            }
            
            if (!apiKeyFile.exists()) {
                log.warn("api-key.txt not found at expected paths")
                return null
            }
            
            def key = apiKeyFile.text?.trim()
            
            if (key) {
                log.info("Auto-discovered API key from api-key.txt")
                return key
            }
            
            log.warn("api-key.txt is empty")
            return null
            
        } catch (Exception e) {
            log.warn("Error reading api-key.txt: ${e.message}")
            return null
        }
    }
    
    /**
     * Refresh the backend URL from settings.xml.
     * Call this if you suspect the server port may have changed.
     */
    void refreshBackendUrl() {
        def newUrl = discoverBackendUrl()
        if (newUrl && newUrl != backendBaseUrl) {
            log.info("Backend URL changed from ${backendBaseUrl} to ${newUrl}")
            backendBaseUrl = newUrl
        }
    }

    /**
     * Fetch report data from the ReportBurster backend.
     * 
     * @param configurationFilePath Path to the configuration XML file (optional, uses default if not provided)
     * @param parameters Map of parameter name to value for parameterized queries
     * @return Map containing reportData, reportColumnNames, executionTimeMillis, isPreview, totalRows
     * @throws Exception if the backend request fails
     */
    Map fetchReportData(String configurationFilePath = null, Map<String, String> parameters = [:]) {
        def configPath = configurationFilePath ?: defaultConfigPath
        
        // Build query string
        def queryParams = ["configurationFilePath=${URLEncoder.encode(configPath, 'UTF-8')}"]
        parameters.each { key, value ->
            queryParams << "${URLEncoder.encode(key, 'UTF-8')}=${URLEncoder.encode(value?.toString() ?: '', 'UTF-8')}"
        }
        
        def url = "${backendBaseUrl}/jobman/reporting/fetch-data?${queryParams.join('&')}"
        log.info("Fetching report data from: ${url}")
        
        try {
            def connection = new URL(url).openConnection() as HttpURLConnection
            connection.requestMethod = 'GET'
            connection.setRequestProperty('Accept', 'application/json')
            
            // Add API key header if available
            if (apiKey) {
                connection.setRequestProperty('X-API-Key', apiKey)
            }
            
            connection.connectTimeout = 30000
            connection.readTimeout = 60000
            
            def responseCode = connection.responseCode
            if (responseCode == 200) {
                def response = connection.inputStream.text
                def result = new JsonSlurper().parseText(response) as Map
                log.info("Fetched ${result.reportData?.size() ?: 0} rows in ${result.executionTimeMillis}ms")
                return result
            } else {
                def errorText = connection.errorStream?.text ?: 'Unknown error'
                log.error("Backend returned ${responseCode}: ${errorText}")
                throw new RuntimeException("Backend error (${responseCode}): ${errorText}")
            }
        } catch (ConnectException e) {
            log.error("Cannot connect to ReportBurster backend at ${backendBaseUrl}")
            throw new RuntimeException("Cannot connect to ReportBurster backend. Is it running at ${backendBaseUrl}?", e)
        }
    }

    /**
     * Get sample data for demo purposes when backend is not available.
     * This can be used for local development/testing.
     */
    Map getSampleData() {
        return [
            reportData: [
                [month: 'Jan', revenue: 45000, expenses: 32000, profit: 13000],
                [month: 'Feb', revenue: 52000, expenses: 35000, profit: 17000],
                [month: 'Mar', revenue: 48000, expenses: 33000, profit: 15000],
                [month: 'Apr', revenue: 61000, expenses: 38000, profit: 23000],
                [month: 'May', revenue: 55000, expenses: 36000, profit: 19000],
                [month: 'Jun', revenue: 67000, expenses: 40000, profit: 27000]
            ],
            reportColumnNames: ['month', 'revenue', 'expenses', 'profit'],
            executionTimeMillis: 0,
            isPreview: true,
            totalRows: 6
        ]
    }
}
