/*
 * Decompiled with CFR 0.152.
 */
package com.sourcekraft.documentburster.common.db;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.sourcekraft.documentburster.common.db.schema.ColumnSchema;
import com.sourcekraft.documentburster.common.db.schema.ForeignKeySchema;
import com.sourcekraft.documentburster.common.db.schema.IndexSchema;
import com.sourcekraft.documentburster.common.db.schema.SchemaInfo;
import com.sourcekraft.documentburster.common.db.schema.TableSchema;
import com.sourcekraft.documentburster.common.settings.Settings;
import com.sourcekraft.documentburster.common.settings.model.DocumentBursterConnectionDatabaseSettings;
import com.sourcekraft.documentburster.common.settings.model.ServerDatabaseSettings;
import com.sourcekraft.documentburster.utils.Utils;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.DriverManager;
import java.sql.ResultSet;
import java.sql.SQLException;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DatabaseSchemaFetcher {
    private static final Logger log = LoggerFactory.getLogger(DatabaseSchemaFetcher.class);
    private final ObjectMapper objectMapper = new ObjectMapper();

    public DatabaseSchemaFetcher() {
        this.objectMapper.enable(SerializationFeature.INDENT_OUTPUT);
    }

    public SchemaInfo fetchSchema(String connectionFilePath) throws Exception {
        Settings settings = new Settings("");
        DocumentBursterConnectionDatabaseSettings dbSettings = settings.loadSettingsConnectionDatabaseByPath(connectionFilePath);
        ServerDatabaseSettings serverSettings = dbSettings.connection.databaseserver;
        if (settings == null || StringUtils.isBlank((CharSequence)serverSettings.type)) {
            throw new IllegalArgumentException("Database settings or type cannot be null or blank.");
        }
        String dbType = serverSettings.type.toLowerCase();
        log.info("Attempting to fetch schema for database type: {}", (Object)dbType);
        Connection connection = null;
        try {
            connection = this.getConnection(serverSettings);
            log.info("Successfully established connection for schema fetching.");
            DatabaseMetaData metaData = connection.getMetaData();
            SchemaInfo schemaInfo = new SchemaInfo();
            this.fetchTablesAndColumns(metaData, schemaInfo, serverSettings);
            log.info("Successfully fetched schema metadata.");
            SchemaInfo schemaInfo2 = schemaInfo;
            return schemaInfo2;
        }
        catch (Exception e) {
            log.error("Failed to fetch database schema for type '{}'. Reason: {}", new Object[]{dbType, e.getMessage(), e});
            throw e;
        }
        finally {
            if (connection != null) {
                try {
                    connection.close();
                    log.debug("Database connection closed.");
                }
                catch (SQLException e) {
                    log.warn("Error closing database connection: {}", (Object)e.getMessage());
                }
            }
        }
    }

    public void saveSchemaToJson(SchemaInfo schemaInfo, String outputPath) throws Exception {
        if (schemaInfo == null) {
            throw new IllegalArgumentException("SchemaInfo object cannot be null.");
        }
        if (StringUtils.isBlank((CharSequence)outputPath)) {
            throw new IllegalArgumentException("Output JSON file path cannot be blank.");
        }
        File outputFile = new File(outputPath);
        File parentDir = outputFile.getParentFile();
        if (parentDir != null && !parentDir.exists()) {
            log.info("Creating parent directory for JSON output: {}", (Object)parentDir.getAbsolutePath());
            if (!parentDir.mkdirs()) {
                throw new IOException("Failed to create parent directory: " + parentDir.getAbsolutePath());
            }
        }
        log.info("Saving fetched schema to JSON file: {}", (Object)outputPath);
        try {
            this.objectMapper.writeValue(outputFile, (Object)schemaInfo);
            log.info("Schema successfully saved to {}", (Object)outputPath);
        }
        catch (Exception e) {
            log.error("Failed to write schema to JSON file '{}': {}", new Object[]{outputPath, e.getMessage(), e});
            throw new Exception("Failed to save schema to JSON.", e);
        }
    }

    public SchemaInfo loadSchemaFromJson(String inputPath) throws Exception {
        if (StringUtils.isBlank((CharSequence)inputPath)) {
            throw new IllegalArgumentException("Input JSON file path cannot be blank.");
        }
        File inputFile = new File(inputPath);
        if (!inputFile.exists() || !inputFile.isFile()) {
            throw new FileNotFoundException("Schema JSON file not found or is not a file: " + inputPath);
        }
        log.info("Loading schema from JSON file: {}", (Object)inputPath);
        try {
            SchemaInfo schemaInfo = (SchemaInfo)this.objectMapper.readValue(inputFile, SchemaInfo.class);
            log.info("Schema successfully loaded from {}", (Object)inputPath);
            return schemaInfo;
        }
        catch (Exception e) {
            log.error("Failed to read or parse schema from JSON file '{}': {}", new Object[]{inputPath, e.getMessage(), e});
            throw new Exception("Failed to load schema from JSON.", e);
        }
    }

    private Connection getConnection(ServerDatabaseSettings settings) throws Exception {
        String dbType = settings.type.toLowerCase();
        settings.ensureDriverAndUrl();
        Connection connection = null;
        String driverClass = this.getDriverClass(dbType);
        String jdbcUrl = this.getJdbcUrl(settings);
        if (driverClass == null || jdbcUrl == null) {
            throw new UnsupportedOperationException("Database type '" + dbType + "' is not supported for connection.");
        }
        try {
            log.debug("Loading JDBC driver: {}", (Object)driverClass);
            Class.forName(driverClass);
            log.info("JDBC driver loaded successfully.");
        }
        catch (ClassNotFoundException e) {
            log.error("JDBC Driver ({}) not found. Ensure it's in the classpath.", (Object)driverClass, (Object)e);
            throw new RuntimeException("JDBC Driver not found: " + driverClass, e);
        }
        try {
            log.debug("Attempting to connect using JDBC URL: {}", (Object)jdbcUrl);
            connection = "sqlite".equals(dbType) ? DriverManager.getConnection(jdbcUrl) : DriverManager.getConnection(jdbcUrl, settings.userid, settings.userpassword);
            return connection;
        }
        catch (SQLException e) {
            log.error("SQL error during connection attempt: {} (SQLState: {}, ErrorCode: {})", new Object[]{e.getMessage(), e.getSQLState(), e.getErrorCode()});
            throw new Exception("Failed to establish database connection.", e);
        }
    }

    private String getDriverClass(String dbType) {
        switch (dbType) {
            case "sqlite": {
                return "org.sqlite.JDBC";
            }
            case "oracle": {
                return "oracle.jdbc.driver.OracleDriver";
            }
            case "sqlserver": {
                return "com.microsoft.sqlserver.jdbc.SQLServerDriver";
            }
            case "postgres": 
            case "postgresql": {
                return "org.postgresql.Driver";
            }
            case "mysql": {
                return "com.mysql.cj.jdbc.Driver";
            }
            case "mariadb": {
                return "org.mariadb.jdbc.Driver";
            }
            case "ibmdb2": {
                return "com.ibm.db2.jcc.DB2Driver";
            }
        }
        log.warn("No JDBC driver class configured for database type: {}", (Object)dbType);
        return null;
    }

    private String getJdbcUrl(ServerDatabaseSettings settings) {
        String dbType = settings.type.toLowerCase();
        String effectiveHost = Utils.getEffectiveHost(settings.host);
        switch (dbType) {
            case "sqlite": {
                if (StringUtils.isBlank((CharSequence)settings.database)) {
                    return null;
                }
                return "jdbc:sqlite:" + settings.database.replace("\\", "/");
            }
            case "oracle": {
                return String.format("jdbc:oracle:thin:@//%s:%s/%s", effectiveHost, settings.port, settings.database);
            }
            case "sqlserver": {
                String sslProps = settings.usessl ? ";encrypt=true;trustServerCertificate=true" : ";encrypt=false";
                return String.format("jdbc:sqlserver://%s:%s;databaseName=%s%s", effectiveHost, settings.port, settings.database, sslProps);
            }
            case "postgres": 
            case "postgresql": {
                String pgSsl = settings.usessl ? "?ssl=true&sslmode=prefer" : "";
                return String.format("jdbc:postgresql://%s:%s/%s%s", effectiveHost, settings.port, settings.database, pgSsl);
            }
            case "mysql": {
                String mysqlSsl = settings.usessl ? "?useSSL=true&requireSSL=true" : "?useSSL=false";
                return String.format("jdbc:mysql://%s:%s/%s%s", effectiveHost, settings.port, settings.database, mysqlSsl);
            }
            case "mariadb": {
                String mariaSsl = settings.usessl ? "?useSSL=true" : "?useSSL=false";
                return String.format("jdbc:mariadb://%s:%s/%s%s", effectiveHost, settings.port, settings.database, mariaSsl);
            }
            case "ibmdb2": {
                String db2Ssl = settings.usessl ? ":sslConnection=true;" : "";
                return String.format("jdbc:db2://%s:%s/%s%s", effectiveHost, settings.port, settings.database, db2Ssl);
            }
        }
        log.warn("Cannot construct JDBC URL for unsupported database type: {}", (Object)dbType);
        return null;
    }

    private void fetchTablesAndColumns(DatabaseMetaData metaData, SchemaInfo schemaInfo, ServerDatabaseSettings settings) throws SQLException {
        String catalog = this.getCatalog(settings, metaData);
        String schemaPattern = this.getSchemaPattern(settings, metaData);
        String tableNamePattern = "%";
        CharSequence[] types = new String[]{"TABLE", "VIEW"};
        log.debug("Fetching tables/views (catalog={}, schemaPattern={}, tableNamePattern={}, types={})...", new Object[]{catalog, schemaPattern, tableNamePattern, String.join((CharSequence)",", types)});
        try (ResultSet tablesResultSet = metaData.getTables(catalog, schemaPattern, tableNamePattern, (String[])types);){
            while (tablesResultSet.next()) {
                String tableType;
                String tableRemarks = tablesResultSet.getString("REMARKS");
                String tableName = tablesResultSet.getString("TABLE_NAME");
                if (this.isSystemTableOrView(tableName, tableType = tablesResultSet.getString("TABLE_TYPE"), settings.type)) {
                    log.debug("Skipping potential system object: {} (Type: {})", (Object)tableName, (Object)tableType);
                    continue;
                }
                log.info("Processing object: {} (Type: {})", (Object)tableName, (Object)tableType);
                TableSchema tableSchema = new TableSchema();
                tableSchema.tableName = tableName;
                tableSchema.tableType = tableType;
                tableSchema.remarks = tableRemarks;
                this.fetchColumnsForTable(metaData, catalog, schemaPattern, tableName, tableSchema);
                if ("TABLE".equalsIgnoreCase(tableType)) {
                    this.fetchPrimaryKeysForTable(metaData, catalog, schemaPattern, tableName, tableSchema);
                }
                if ("TABLE".equalsIgnoreCase(tableType)) {
                    this.fetchForeignKeysForTable(metaData, catalog, schemaPattern, tableName, tableSchema);
                }
                if ("TABLE".equalsIgnoreCase(tableType)) {
                    this.fetchIndexesForTable(metaData, catalog, schemaPattern, tableName, tableSchema);
                }
                schemaInfo.tables.add(tableSchema);
            }
        }
    }

    protected String getCatalog(ServerDatabaseSettings settings, DatabaseMetaData metaData) throws SQLException {
        String t;
        switch (t = settings.type == null ? "" : settings.type.toLowerCase()) {
            case "mysql": 
            case "mariadb": {
                return settings.database;
            }
            case "sqlserver": {
                return settings.database;
            }
        }
        return null;
    }

    protected String getSchemaPattern(ServerDatabaseSettings settings, DatabaseMetaData metaData) throws SQLException {
        String t;
        switch (t = settings.type == null ? "" : settings.type.toLowerCase()) {
            case "oracle": {
                return StringUtils.isNotBlank((CharSequence)settings.userid) ? settings.userid.toUpperCase() : null;
            }
            case "postgresql": 
            case "postgres": {
                return "public";
            }
            case "sqlserver": {
                return "dbo";
            }
            case "mysql": 
            case "mariadb": {
                return null;
            }
            case "ibmdb2": 
            case "db2": {
                return StringUtils.isNotBlank((CharSequence)settings.userid) ? settings.userid.toUpperCase() : null;
            }
        }
        return null;
    }

    private boolean isSystemTableOrView(String name, String type, String dbType) {
        String lowerCaseName = name.toLowerCase();
        if (lowerCaseName.startsWith("information_schema.") || lowerCaseName.equals("information_schema")) {
            return true;
        }
        if (lowerCaseName.startsWith("pg_catalog.") || lowerCaseName.equals("pg_catalog")) {
            return true;
        }
        if (lowerCaseName.startsWith("sys.") || lowerCaseName.equals("sys")) {
            return true;
        }
        if (lowerCaseName.startsWith("system.") || lowerCaseName.equals("system")) {
            return true;
        }
        if (lowerCaseName.equals("mysql") || lowerCaseName.startsWith("mysql.")) {
            return true;
        }
        if (lowerCaseName.equals("performance_schema") || lowerCaseName.startsWith("performance_schema.")) {
            return true;
        }
        switch (dbType) {
            case "sqlite": {
                return lowerCaseName.startsWith("sqlite_");
            }
            case "postgresql": {
                return lowerCaseName.startsWith("pg_");
            }
            case "oracle": {
                return lowerCaseName.startsWith("all_") || lowerCaseName.startsWith("user_") || lowerCaseName.startsWith("dba_") || lowerCaseName.startsWith("v$") || lowerCaseName.startsWith("gv$");
            }
            case "sqlserver": {
                return false;
            }
            case "ibmdb2": {
                return lowerCaseName.startsWith("sysibm_") || lowerCaseName.startsWith("syscat_") || lowerCaseName.startsWith("sysstat_");
            }
            case "mysql": 
            case "mariadb": {
                return false;
            }
        }
        return false;
    }

    private void fetchColumnsForTable(DatabaseMetaData metaData, String catalog, String schemaPattern, String tableName, TableSchema tableSchema) throws SQLException {
        log.debug("Fetching columns for table/view: {}", (Object)tableName);
        try (ResultSet columnsResultSet = metaData.getColumns(catalog, schemaPattern, tableName, "%");){
            while (columnsResultSet.next()) {
                ColumnSchema columnSchema = new ColumnSchema();
                columnSchema.remarks = columnsResultSet.getString("REMARKS");
                columnSchema.defaultValue = columnsResultSet.getString("COLUMN_DEF");
                columnSchema.columnName = columnsResultSet.getString("COLUMN_NAME");
                columnSchema.dataType = columnsResultSet.getInt("DATA_TYPE");
                columnSchema.typeName = columnsResultSet.getString("TYPE_NAME");
                columnSchema.columnSize = columnsResultSet.getInt("COLUMN_SIZE");
                columnSchema.decimalDigits = DatabaseSchemaFetcher.getNullableInt(columnsResultSet, "DECIMAL_DIGITS");
                columnSchema.numPrecRadix = DatabaseSchemaFetcher.getNullableInt(columnsResultSet, "NUM_PREC_RADIX");
                columnSchema.isNullable = "YES".equalsIgnoreCase(columnsResultSet.getString("IS_NULLABLE"));
                tableSchema.columns.add(columnSchema);
                log.trace("  Added column: {} (Type: {}, Size: {}, DecDigits: {}, PrecRadix: {}, Nullable: {}, Remarks: '{}', Default: '{}')", new Object[]{columnSchema.columnName, columnSchema.typeName, columnSchema.columnSize, columnSchema.decimalDigits, columnSchema.numPrecRadix, columnSchema.isNullable, columnSchema.remarks, columnSchema.defaultValue});
            }
        }
    }

    private static Integer getNullableInt(ResultSet rs, String column) throws SQLException {
        int v = rs.getInt(column);
        return rs.wasNull() ? null : Integer.valueOf(v);
    }

    private void fetchPrimaryKeysForTable(DatabaseMetaData metaData, String catalog, String schemaPattern, String tableName, TableSchema tableSchema) throws SQLException {
        log.debug("Fetching primary keys for table: {}", (Object)tableName);
        try (ResultSet pkResultSet = metaData.getPrimaryKeys(catalog, schemaPattern, tableName);){
            while (pkResultSet.next()) {
                String pkColumnName = pkResultSet.getString("COLUMN_NAME");
                tableSchema.primaryKeyColumns.add(pkColumnName);
                log.trace("  Added primary key column: {}", (Object)pkColumnName);
            }
        }
    }

    private void fetchForeignKeysForTable(DatabaseMetaData metaData, String catalog, String schemaPattern, String tableName, TableSchema tableSchema) throws SQLException {
        log.debug("Fetching foreign keys for table: {}", (Object)tableName);
        try (ResultSet fkResultSet = metaData.getImportedKeys(catalog, schemaPattern, tableName);){
            while (fkResultSet.next()) {
                ForeignKeySchema fkSchema = new ForeignKeySchema();
                fkSchema.fkName = fkResultSet.getString("FK_NAME");
                fkSchema.fkColumnName = fkResultSet.getString("FKCOLUMN_NAME");
                fkSchema.pkTableName = fkResultSet.getString("PKTABLE_NAME");
                fkSchema.pkColumnName = fkResultSet.getString("PKCOLUMN_NAME");
                tableSchema.foreignKeys.add(fkSchema);
                log.trace("  Added foreign key: {} ({}) -> {}({})", new Object[]{fkSchema.fkName, fkSchema.fkColumnName, fkSchema.pkTableName, fkSchema.pkColumnName});
            }
        }
    }

    private void fetchIndexesForTable(DatabaseMetaData metaData, String catalog, String schemaPattern, String tableName, TableSchema tableSchema) throws SQLException {
        log.debug("Fetching indexes for table: {}", (Object)tableName);
        try (ResultSet rs = metaData.getIndexInfo(catalog, schemaPattern, tableName, false, true);){
            this.processIndexRows(rs, tableSchema);
            return;
        }
        catch (SQLException firstEx) {
            if (!DatabaseSchemaFetcher.isOracle17068(firstEx, metaData)) {
                log.warn("Could not retrieve index info for table '{}' (catalog={}, schema={}): {}", new Object[]{tableName, catalog, schemaPattern, firstEx.getMessage()});
                return;
            }
            log.debug("Oracle ORA-17068 on getIndexInfo for '{}'. Applying fallbacks...", (Object)tableName);
            try (ResultSet rs22 = metaData.getIndexInfo(catalog, schemaPattern, tableName, false, false);){
                this.processIndexRows(rs22, tableSchema);
                log.debug("Oracle fallback (approximate=false) succeeded for '{}'", (Object)tableName);
                return;
            }
            catch (SQLException rs22) {
                try (ResultSet rs32 = metaData.getIndexInfo(catalog, null, tableName, false, true);){
                    this.processIndexRows(rs32, tableSchema);
                    log.debug("Oracle fallback (schema=null) succeeded for '{}'", (Object)tableName);
                    return;
                }
                catch (SQLException rs32) {
                    String quotedTable = "\"" + tableName + "\"";
                    try (ResultSet rs4 = metaData.getIndexInfo(catalog, schemaPattern, quotedTable, false, true);){
                        this.processIndexRows(rs4, tableSchema);
                        log.debug("Oracle fallback (quoted table name) succeeded for '{}'", (Object)tableName);
                        return;
                    }
                    catch (SQLException sQLException) {
                        log.warn("Could not retrieve index info for table '{}' (catalog={}, schema={}): {}", new Object[]{tableName, catalog, schemaPattern, firstEx.getMessage()});
                        return;
                    }
                }
            }
        }
    }

    private static boolean isOracle17068(SQLException ex, DatabaseMetaData metaData) {
        try {
            String product = metaData.getDatabaseProductName();
            boolean isOracle = product != null && product.toLowerCase().contains("oracle");
            String msg = ex.getMessage() == null ? "" : ex.getMessage();
            return isOracle && (msg.contains("ORA-17068") || "17068".equals(ex.getSQLState()));
        }
        catch (SQLException e) {
            return false;
        }
    }

    private void processIndexRows(ResultSet indexResultSet, TableSchema tableSchema) throws SQLException {
        while (indexResultSet.next()) {
            String indexName = indexResultSet.getString("INDEX_NAME");
            short indexType = indexResultSet.getShort("TYPE");
            String columnName = indexResultSet.getString("COLUMN_NAME");
            boolean nonUnique = indexResultSet.getBoolean("NON_UNIQUE");
            short ordinalPosition = indexResultSet.getShort("ORDINAL_POSITION");
            if (indexType == 0 || indexName == null || columnName == null && indexName != null) continue;
            IndexSchema indexSchema = new IndexSchema();
            indexSchema.indexName = indexName;
            indexSchema.columnName = columnName;
            indexSchema.isUnique = !nonUnique;
            indexSchema.type = indexType;
            indexSchema.ordinalPosition = ordinalPosition;
            tableSchema.indexes.add(indexSchema);
        }
    }
}

