/*
 * Decompiled with CFR 0.152.
 */
package com.sourcekraft.documentburster.common.db.northwind;

import com.sourcekraft.documentburster.common.db.northwind.NorthwindDataGenerator;
import com.sourcekraft.documentburster.utils.Utils;
import jakarta.persistence.EntityManager;
import jakarta.persistence.EntityManagerFactory;
import jakarta.persistence.Persistence;
import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.FileAttribute;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.ResultSet;
import java.sql.Statement;
import java.time.Duration;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import net.jodah.failsafe.Failsafe;
import net.jodah.failsafe.Policy;
import net.jodah.failsafe.RetryPolicy;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.zeroturnaround.exec.ProcessExecutor;
import org.zeroturnaround.exec.ProcessResult;

public class NorthwindManager
implements AutoCloseable {
    private static final Logger log = LoggerFactory.getLogger(NorthwindManager.class);
    private static final String DOCKER_COMPOSE_FILENAME = "docker-compose.yml";
    private static final String MARKER_FILENAME = ".northwind_initialized";
    private final Map<DatabaseVendor, Boolean> runningDatabases = new HashMap<DatabaseVendor, Boolean>();
    private final Map<DatabaseVendor, Path> activeDataPaths = new HashMap<DatabaseVendor, Path>();
    private final Map<DatabaseVendor, Integer> activeHostPorts = new HashMap<DatabaseVendor, Integer>();
    private final String baseDataPath = Utils.getDbFolderPath();
    private final String dockerComposeFilePath;
    private File sqliteDbFile;

    public NorthwindManager() throws Exception {
        Path dbTemplatePath = Paths.get(this.baseDataPath, new String[0]);
        this.dockerComposeFilePath = dbTemplatePath.resolve(DOCKER_COMPOSE_FILENAME).toString();
        for (DatabaseVendor vendor : DatabaseVendor.values()) {
            this.runningDatabases.put(vendor, false);
        }
    }

    public void startDatabase(DatabaseVendor vendor, String customHostPath, Integer customHostPort) throws Exception {
        boolean needsInitialization;
        Path hostDataPath = customHostPath != null && !customHostPath.trim().isEmpty() ? Paths.get(customHostPath, new String[0]).toAbsolutePath() : this.getDefaultHostDataPath(vendor);
        Integer hostPortToUse = customHostPort != null ? customHostPort : this.getDefaultHostPort(vendor);
        log.info("[SQL_SERVER_DEBUG] Starting database: vendor={}, hostPort={}, hostDataPath={}", new Object[]{vendor, hostPortToUse, hostDataPath});
        if (vendor == DatabaseVendor.SQLITE) {
            this.sqliteDbFile = hostDataPath.resolve("northwind.db").toFile();
            needsInitialization = !this.sqliteDbFile.exists() || this.sqliteDbFile.length() == 0L;
        } else {
            Path markerFile = hostDataPath.resolve(MARKER_FILENAME);
            needsInitialization = !Files.exists(markerFile, new LinkOption[0]);
        }
        this.activeDataPaths.put(vendor, hostDataPath);
        if (hostPortToUse != null) {
            this.activeHostPorts.put(vendor, hostPortToUse);
        }
        if (vendor != DatabaseVendor.SQLITE) {
            this.startDatabaseWithDockerCompose(vendor, hostPortToUse);
        } else {
            this.setupSQLite(hostDataPath);
        }
        this.runningDatabases.put(vendor, true);
        if (needsInitialization) {
            this.initializeDatabaseWithGenerator(vendor, hostDataPath);
        }
        log.info("Northwind DB Started: {} | Data Path: {} | JDBC URL: {}", new Object[]{vendor, this.getActualDataPath(vendor), this.getJdbcUrl(vendor)});
    }

    public void startDatabase(DatabaseVendor vendor, String customHostPath) throws Exception {
        this.startDatabase(vendor, customHostPath, null);
    }

    private void startDatabaseWithDockerCompose(DatabaseVendor vendor, Integer hostPort) throws Exception {
        String serviceName = vendor.getDockerComposeServiceName();
        Path workingDir = Paths.get(this.dockerComposeFilePath, new String[0]).getParent();
        ArrayList<String> command = new ArrayList<String>();
        command.add("docker");
        command.add("compose");
        command.add("-f");
        command.add(DOCKER_COMPOSE_FILENAME);
        command.add("up");
        command.add("-d");
        command.add("--remove-orphans");
        command.add(serviceName);
        ProcessExecutor executor = new ProcessExecutor().command(command).directory(workingDir.toFile()).readOutput(true).timeout(vendor.getDefaultTimeout().toSeconds(), TimeUnit.SECONDS);
        if (hostPort != null) {
            HashMap<String, String> environment = new HashMap<String, String>(System.getenv());
            String portEnvVar = vendor.name();
            portEnvVar = portEnvVar + "_PORT";
            environment.put(portEnvVar, hostPort.toString());
            executor.environment(environment);
        }
        log.debug("Executing: {}", (Object)String.join((CharSequence)" ", command));
        log.info("[SQL_SERVER_DEBUG] Executing docker-compose command: {}", (Object)String.join((CharSequence)" ", command));
        log.info("[SQL_SERVER_DEBUG] Working directory: {}", (Object)workingDir);
        if (hostPort != null) {
            log.info("[SQL_SERVER_DEBUG] Environment variable {}={}", (Object)(vendor.name() + "_PORT"), (Object)hostPort);
        }
        ProcessResult result = executor.execute();
        log.info("[SQL_SERVER_DEBUG] Docker compose exit code: {}, output: {}", (Object)result.getExitValue(), (Object)result.outputUTF8());
        if (result.getExitValue() != 0) {
            throw new IOException("Docker Compose command failed with exit code " + result.getExitValue() + " for service " + serviceName);
        }
        if (vendor == DatabaseVendor.SQLSERVER) {
            log.info("[SQL_SERVER_DEBUG] SQL Server started, creating Northwind database...");
            this.ensureSqlServerDatabaseExists(vendor, hostPort);
        }
        this.waitForDatabaseToBeReady(vendor, hostPort);
    }

    public void stopDatabase(DatabaseVendor vendor) throws Exception {
        if (vendor == DatabaseVendor.SQLITE) {
            this.sqliteDbFile = null;
        } else {
            String serviceName = vendor.getDockerComposeServiceName();
            Path workingDir = Paths.get(this.dockerComposeFilePath, new String[0]).getParent();
            ArrayList<String> command = new ArrayList<String>();
            command.add("docker");
            command.add("compose");
            command.add("-f");
            command.add(DOCKER_COMPOSE_FILENAME);
            command.add("stop");
            command.add(serviceName);
            new ProcessExecutor().command(command).directory(workingDir.toFile()).execute();
        }
        this.runningDatabases.put(vendor, false);
        this.activeDataPaths.remove((Object)vendor);
        this.activeHostPorts.remove((Object)vendor);
        log.info("Service {} stopped.", (Object)vendor);
    }

    private void waitForDatabaseToBeReady(DatabaseVendor vendor, Integer hostPort) throws Exception {
        int port = hostPort != null ? hostPort : this.getDefaultHostPort(vendor);
        String host = Utils.getEffectiveHost("localhost");
        log.info("[SQL_SERVER_DEBUG] waitForDatabaseToBeReady: vendor={}, host={}, port={}", new Object[]{vendor, host, port});
        RetryPolicy retryPolicy = (RetryPolicy)((RetryPolicy)new RetryPolicy().handle(Throwable.class)).withDelay(Duration.ofSeconds(5L)).withMaxDuration(vendor.getDefaultTimeout()).withMaxRetries(-1).onRetry(e -> log.debug("Connection to {} on port {} failed, retrying...", (Object)vendor, (Object)port)).onFailure(e -> log.error("Failed to connect to {} after multiple retries.", (Object)vendor));
        Failsafe.with((Policy[])new RetryPolicy[]{retryPolicy}).run(() -> {
            log.info("[SQL_SERVER_DEBUG] Retry attempt starting for {}:{}", (Object)vendor, (Object)port);
            if (!Utils.isPortOpen(host, port, 2000)) {
                throw new IOException("Port " + port + " is not open.");
            }
            log.info("[SQL_SERVER_DEBUG] Port {} is open, attempting JDBC connection", (Object)port);
            String jdbcUrl = this.getJdbcUrl(vendor);
            log.info("Attempting JDBC connection with URL: {}", (Object)jdbcUrl);
            log.info("[SQL_SERVER_DEBUG] DriverManager.getConnection(url={}, user={}, pass=***)", (Object)jdbcUrl, (Object)this.getUsername(vendor));
            DriverManager.setLoginTimeout(5);
            try (Connection connection = DriverManager.getConnection(jdbcUrl, this.getUsername(vendor), this.getPassword(vendor));){
                log.info("[SQL_SERVER_DEBUG] Connection established, checking validity");
                if (!connection.isValid(5)) {
                    log.info("[SQL_SERVER_DEBUG] Connection is NOT valid");
                    throw new IOException("JDBC connection is not valid.");
                }
                log.info("[SQL_SERVER_DEBUG] Connection is valid, database is ready");
            }
        });
        log.info("[SQL_SERVER_DEBUG] waitForDatabaseToBeReady completed successfully");
    }

    private void ensureSqlServerDatabaseExists(DatabaseVendor vendor, Integer hostPort) throws Exception {
        int port = hostPort != null ? hostPort : this.getDefaultHostPort(vendor);
        String host = Utils.getEffectiveHost("localhost");
        String masterUrl = "jdbc:sqlserver://" + host + ":" + port + ";databaseName=master;encrypt=false;trustServerCertificate=true";
        String dbName = vendor.getDefaultDbName();
        log.info("[SQL_SERVER_DEBUG] Waiting for SQL Server to accept connections on master DB...");
        RetryPolicy retryPolicy = (RetryPolicy)((RetryPolicy)new RetryPolicy().handle(Throwable.class)).withDelay(Duration.ofSeconds(5L)).withMaxDuration(Duration.ofMinutes(5L)).withMaxRetries(-1).onRetry(e -> {
            Throwable cause = e.getLastFailure();
            log.debug("[SQL_SERVER_DEBUG] Waiting for SQL Server (attempt #{}): {}", (Object)e.getAttemptCount(), (Object)(cause != null ? cause.getMessage() : "unknown"));
        }).onFailure(e -> log.error("[SQL_SERVER_DEBUG] Failed to connect to SQL Server master", e.getFailure()));
        Failsafe.with((Policy[])new RetryPolicy[]{retryPolicy}).run(() -> {
            if (!Utils.isPortOpen(host, port, 2000)) {
                throw new IOException("Port " + port + " not open");
            }
            log.debug("[SQL_SERVER_DEBUG] Port {} open, attempting connection to master", (Object)port);
            DriverManager.setLoginTimeout(5);
            try (Connection conn = DriverManager.getConnection(masterUrl, this.getUsername(vendor), this.getPassword(vendor));){
                if (!conn.isValid(5)) {
                    throw new IOException("Connection to master not valid");
                }
                log.info("[SQL_SERVER_DEBUG] Connected to master DB, checking if {} exists", (Object)dbName);
                try (Statement stmt = conn.createStatement();){
                    ResultSet rs = stmt.executeQuery("SELECT database_id FROM sys.databases WHERE name = '" + dbName + "'");
                    if (!rs.next()) {
                        log.info("[SQL_SERVER_DEBUG] Database {} does not exist, creating it...", (Object)dbName);
                        stmt.execute("CREATE DATABASE [" + dbName + "]");
                        Thread.sleep(2000L);
                        log.info("[SQL_SERVER_DEBUG] Database {} created successfully", (Object)dbName);
                    } else {
                        log.info("[SQL_SERVER_DEBUG] Database {} already exists", (Object)dbName);
                    }
                }
            }
        });
        log.info("[SQL_SERVER_DEBUG] ensureSqlServerDatabaseExists() completed");
    }

    public Connection getConnection(DatabaseVendor vendor) throws Exception {
        return DriverManager.getConnection(this.getJdbcUrl(vendor), this.getUsername(vendor), this.getPassword(vendor));
    }

    public String getJdbcUrl(DatabaseVendor vendor) {
        int port = this.activeHostPorts.getOrDefault((Object)vendor, this.getDefaultHostPort(vendor));
        String dbName = vendor.getDefaultDbName();
        String host = Utils.getEffectiveHost("localhost");
        switch (vendor) {
            case POSTGRES: {
                return "jdbc:postgresql://" + host + ":" + port + "/" + dbName;
            }
            case MYSQL: {
                return "jdbc:mysql://" + host + ":" + port + "/" + dbName + "?useSSL=false&allowPublicKeyRetrieval=true";
            }
            case MARIADB: {
                return "jdbc:mariadb://" + host + ":" + port + "/" + dbName + "?useSSL=false&allowPublicKeyRetrieval=true";
            }
            case SQLITE: {
                return "jdbc:sqlite:" + this.getActualDataPath(vendor).resolve("northwind.db").toString();
            }
            case SQLSERVER: {
                String url = "jdbc:sqlserver://" + host + ":" + port + ";databaseName=" + dbName + ";encrypt=false";
                log.info("[SQL_SERVER_DEBUG] Constructed SQL Server URL: {}", (Object)url);
                return url;
            }
            case ORACLE: {
                return "jdbc:oracle:thin:@//" + host + ":" + port + "/" + dbName;
            }
            case DB2: {
                return "jdbc:db2://" + host + ":" + port + "/" + dbName;
            }
        }
        throw new IllegalArgumentException("Unsupported database vendor: " + (Object)((Object)vendor));
    }

    private void setupSQLite(Path hostDataPath) throws IOException {
        this.sqliteDbFile = hostDataPath.resolve("northwind.db").toFile();
        Files.createDirectories(this.sqliteDbFile.getParentFile().toPath(), new FileAttribute[0]);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void initializeDatabaseWithGenerator(DatabaseVendor vendor, Path hostDataPath) throws Exception {
        log.info("[SQL_SERVER_DEBUG] Starting JPA initialization for {}", (Object)vendor);
        log.info("[SQL_SERVER_DEBUG] Persistence unit: northwind-{}", (Object)vendor.name().toLowerCase());
        EntityManagerFactory emf = null;
        EntityManager em = null;
        String persistenceUnitName = "northwind-" + vendor.name().toLowerCase();
        try {
            Map<String, String> props = this.getJpaProperties(vendor, true);
            if (vendor == DatabaseVendor.DB2) {
                props.putIfAbsent("hibernate.default_schema", "DB2INST1");
            }
            log.info("[SQL_SERVER_DEBUG] JPA properties: url={}, user={}", (Object)props.get("jakarta.persistence.jdbc.url"), (Object)props.get("jakarta.persistence.jdbc.user"));
            log.info("[SQL_SERVER_DEBUG] Creating EntityManagerFactory");
            RetryPolicy emfRetry = ((RetryPolicy)new RetryPolicy().handle(Throwable.class)).withDelay(Duration.ofSeconds(5L)).withMaxRetries(5).onFailedAttempt(ev -> {
                Throwable t = ev.getLastFailure();
                log.warn("EMF creation failed (attempt {} of {}): {}", new Object[]{ev.getAttemptCount(), 5, t != null ? t.getMessage() : "unknown"});
            });
            emf = (EntityManagerFactory)Failsafe.with((Policy[])new RetryPolicy[]{emfRetry}).get(() -> Persistence.createEntityManagerFactory((String)persistenceUnitName, (Map)props));
            log.info("[SQL_SERVER_DEBUG] Creating EntityManager");
            em = emf.createEntityManager();
            log.info("[SQL_SERVER_DEBUG] Running NorthwindDataGenerator");
            NorthwindDataGenerator generator = new NorthwindDataGenerator(em);
            generator.generateAll();
            log.info("[SQL_SERVER_DEBUG] Data generation complete, creating marker file");
            if (vendor != DatabaseVendor.SQLITE) {
                Path markerFilePath = hostDataPath.resolve(MARKER_FILENAME);
                Files.createDirectories(hostDataPath, new FileAttribute[0]);
                Files.createFile(markerFilePath, new FileAttribute[0]);
                log.info("[SQL_SERVER_DEBUG] Marker file created: {}", (Object)markerFilePath);
            }
        }
        finally {
            if (em != null && em.isOpen()) {
                em.close();
            }
            if (emf != null && emf.isOpen()) {
                emf.close();
            }
        }
    }

    public Map<String, String> getJpaProperties(DatabaseVendor vendor, boolean forDdl) {
        HashMap<String, String> properties = new HashMap<String, String>();
        properties.put("jakarta.persistence.jdbc.url", this.getJdbcUrl(vendor));
        properties.put("jakarta.persistence.jdbc.user", this.getUsername(vendor));
        properties.put("jakarta.persistence.jdbc.password", this.getPassword(vendor));
        if (forDdl) {
            properties.put("jakarta.persistence.schema-generation.database.action", "drop-and-create");
            properties.put("hibernate.hbm2ddl.auto", "create-drop");
        }
        return properties;
    }

    public String getUsername(DatabaseVendor vendor) {
        return vendor.getDefaultUser();
    }

    public String getPassword(DatabaseVendor vendor) {
        return vendor.getDefaultPassword();
    }

    public boolean isRunning(DatabaseVendor vendor) {
        return this.runningDatabases.getOrDefault((Object)vendor, false);
    }

    public Path getActualDataPath(DatabaseVendor vendor) {
        return this.activeDataPaths.get((Object)vendor);
    }

    public Integer getActualHostPort(DatabaseVendor vendor) {
        return this.activeHostPorts.get((Object)vendor);
    }

    private Path getDefaultHostDataPath(DatabaseVendor vendor) {
        String subDirName = "sample-northwind-" + vendor.name().toLowerCase().replace("_", "-");
        return Paths.get(this.baseDataPath, subDirName);
    }

    private Integer getDefaultHostPort(DatabaseVendor vendor) {
        return vendor.getContainerPort();
    }

    @Override
    public void close() throws Exception {
        for (DatabaseVendor vendor : DatabaseVendor.values()) {
            if (!this.isRunning(vendor)) continue;
            this.stopDatabase(vendor);
        }
    }

    public String getDatabaseVersion(DatabaseVendor vendor) throws Exception {
        try (Connection conn = DriverManager.getConnection(this.getJdbcUrl(vendor), this.getUsername(vendor), this.getPassword(vendor));){
            String string = conn.getMetaData().getDatabaseProductVersion();
            return string;
        }
    }

    public boolean testDatabaseConnection(DatabaseVendor vendor) throws Exception {
        try (Connection conn = DriverManager.getConnection(this.getJdbcUrl(vendor), this.getUsername(vendor), this.getPassword(vendor));){
            boolean bl = conn.isValid(2);
            return bl;
        }
    }

    public List<String> getTableNames(DatabaseVendor vendor) throws Exception {
        ArrayList<String> tableNames = new ArrayList<String>();
        try (Connection conn = DriverManager.getConnection(this.getJdbcUrl(vendor), this.getUsername(vendor), this.getPassword(vendor));
             ResultSet rs = conn.getMetaData().getTables(null, null, "%", new String[]{"TABLE"});){
            while (rs.next()) {
                tableNames.add(rs.getString("TABLE_NAME"));
            }
        }
        return tableNames;
    }

    public int getTableCount(DatabaseVendor vendor, String tableName) throws Exception {
        try (Connection conn = DriverManager.getConnection(this.getJdbcUrl(vendor), this.getUsername(vendor), this.getPassword(vendor));
             Statement stmt = conn.createStatement();
             ResultSet rs = stmt.executeQuery("SELECT COUNT(*) FROM \"" + tableName + "\"");){
            if (rs.next()) {
                int n = rs.getInt(1);
                return n;
            }
        }
        return 0;
    }

    public static enum DatabaseVendor {
        POSTGRES(5432, "postgres", "postgres", "Northwind", Duration.ofMinutes(30L)),
        MYSQL(3306, "root", "password", "Northwind", Duration.ofMinutes(30L)),
        MARIADB(3307, "root", "password", "Northwind", Duration.ofMinutes(30L)),
        SQLITE(0, null, null, "Northwind", Duration.ofMinutes(10L)),
        SQLSERVER(1433, "sa", "Password123!", "Northwind", Duration.ofMinutes(60L)),
        ORACLE(1521, "oracle", "oracle", "XEPDB1", Duration.ofMinutes(60L)),
        DB2(50000, "db2inst1", "password", "NORTHWND", Duration.ofMinutes(60L));

        private final int containerPort;
        private final String defaultUser;
        private final String defaultPassword;
        private final String defaultDbName;
        private final Duration defaultTimeout;

        private DatabaseVendor(int containerPort, String defaultUser, String defaultPassword, String defaultDbName, Duration defaultTimeout) {
            this.containerPort = containerPort;
            this.defaultUser = defaultUser;
            this.defaultPassword = defaultPassword;
            this.defaultDbName = defaultDbName;
            this.defaultTimeout = defaultTimeout;
        }

        public int getContainerPort() {
            return this.containerPort;
        }

        public String getDefaultUser() {
            return this.defaultUser;
        }

        public String getDefaultPassword() {
            return this.defaultPassword;
        }

        public String getDefaultDbName() {
            return this.defaultDbName;
        }

        public Duration getDefaultTimeout() {
            return this.defaultTimeout;
        }

        public String getDockerComposeServiceName() {
            if (this == SQLSERVER) {
                return "sqlserver";
            }
            return this.name().toLowerCase();
        }
    }
}

