/*
 * Decompiled with CFR 0.152.
 */
package com.sourcekraft.documentburster.common.db;

import com.sourcekraft.documentburster.common.settings.Settings;
import com.sourcekraft.documentburster.common.settings.model.ConnectionDatabaseSettings;
import com.sourcekraft.documentburster.common.settings.model.DocumentBursterConnectionDatabaseSettings;
import com.sourcekraft.documentburster.common.settings.model.ServerDatabaseSettings;
import com.zaxxer.hikari.HikariConfig;
import com.zaxxer.hikari.HikariDataSource;
import java.sql.Connection;
import java.sql.DriverManager;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import javax.sql.DataSource;
import org.apache.commons.lang3.StringUtils;
import org.jdbi.v3.core.Jdbi;
import org.jdbi.v3.core.statement.Query;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DatabaseConnectionManager {
    private static final Logger log = LoggerFactory.getLogger(DatabaseConnectionManager.class);
    private final Settings settings;
    private final Map<String, HikariDataSource> dataSourcePools = new ConcurrentHashMap<String, HikariDataSource>();
    private final Map<String, Jdbi> jdbiInstances = new ConcurrentHashMap<String, Jdbi>();
    private final Map<String, ConnectionDatabaseSettings> loadedConnectionDetails = new ConcurrentHashMap<String, ConnectionDatabaseSettings>();

    public DatabaseConnectionManager(Settings settings) {
        this.settings = settings;
        log.debug("DatabaseConnectionManager initialized.");
    }

    public Settings getSettings() {
        return this.settings;
    }

    private String getDriverClass(String dbType) {
        switch (dbType.toLowerCase()) {
            case "sqlite": {
                return "org.sqlite.JDBC";
            }
            case "oracle": {
                return "oracle.jdbc.driver.OracleDriver";
            }
            case "sqlserver": {
                return "com.microsoft.sqlserver.jdbc.SQLServerDriver";
            }
            case "postgres": 
            case "postgresql": {
                return "org.postgresql.Driver";
            }
            case "mysql": {
                return "com.mysql.cj.jdbc.Driver";
            }
            case "mariadb": {
                return "org.mariadb.jdbc.Driver";
            }
            case "ibmdb2": {
                return "com.ibm.db2.jcc.DB2Driver";
            }
        }
        log.warn("No JDBC driver class configured for database type: {}", (Object)dbType);
        return null;
    }

    public Connection getJdbcConnection(String connectionCode) throws Exception {
        String cCode = "";
        cCode = !StringUtils.isBlank((CharSequence)connectionCode) ? connectionCode : this.settings.getReportingPrimaryDatabaseConnectionCode();
        ConnectionDatabaseSettings settings = this.getConnectionSettings(cCode);
        settings.databaseserver.ensureDriverAndUrl();
        String driverClass = this.getDriverClass(settings.databaseserver.type);
        Class.forName(driverClass);
        return DriverManager.getConnection(settings.databaseserver.url, settings.databaseserver.userid, settings.databaseserver.userpassword);
    }

    private synchronized ConnectionDatabaseSettings getConnectionSettings(String connectionCode) throws Exception {
        String primaryCode = this.settings.getReportingPrimaryDatabaseConnectionCode();
        if (StringUtils.isNotBlank((CharSequence)primaryCode) && primaryCode.equals(connectionCode) && this.settings.connectionDatabaseSettings != null) {
            ConnectionDatabaseSettings c = this.settings.connectionDatabaseSettings.connection;
            if (c != null && c.databaseserver != null) {
                c.databaseserver.ensureDriverAndUrl();
            }
            return c;
        }
        DocumentBursterConnectionDatabaseSettings dbSettings = this.settings.loadSettingsConnectionDatabase(connectionCode);
        if (dbSettings != null && dbSettings.connection != null && dbSettings.connection.databaseserver != null) {
            dbSettings.connection.databaseserver.ensureDriverAndUrl();
        }
        return dbSettings.connection;
    }

    public synchronized DataSource getDataSource(String connectionCode) throws Exception {
        log.trace("Entering getDataSource for code: {}", (Object)connectionCode);
        if (StringUtils.isBlank((CharSequence)connectionCode)) {
            log.error("getDataSource called with blank connection code.");
            throw new IllegalArgumentException("Connection code cannot be blank.");
        }
        if (this.dataSourcePools.containsKey(connectionCode)) {
            log.debug("Cache hit for DataSource: {}", (Object)connectionCode);
            log.trace("Exiting getDataSource (cached) for code: {}", (Object)connectionCode);
            return (DataSource)this.dataSourcePools.get(connectionCode);
        }
        log.debug("Cache miss for DataSource: {}. Creating new pool.", (Object)connectionCode);
        ConnectionDatabaseSettings connSettings = this.getConnectionSettings(connectionCode);
        if (connSettings == null || connSettings.databaseserver == null) {
            log.error("Retrieved ConnectionDatabaseSettings or its databaseserver field is null for code: {}", (Object)connectionCode);
            throw new Exception("Failed to retrieve valid database server settings for code: " + connectionCode);
        }
        connSettings.databaseserver.ensureDriverAndUrl();
        log.trace("Creating HikariConfig for code: {}", (Object)connectionCode);
        HikariConfig config = new HikariConfig();
        if (StringUtils.isBlank((CharSequence)connSettings.databaseserver.driver)) {
            connSettings.databaseserver.driver = this.getDriverClass(connSettings.databaseserver.type);
            if (StringUtils.isBlank((CharSequence)connSettings.databaseserver.driver)) {
                throw new Exception("Database driver not specified and could not be determined for type: " + connSettings.databaseserver.type);
            }
        }
        config.setDriverClassName(connSettings.databaseserver.driver);
        config.setJdbcUrl(connSettings.databaseserver.url);
        config.setUsername(connSettings.databaseserver.userid);
        config.setPassword(connSettings.databaseserver.userpassword);
        log.trace("HikariConfig: Driver={}, URL={}, User={}, Password provided={}", new Object[]{connSettings.databaseserver.driver, connSettings.databaseserver.url, connSettings.databaseserver.userid, StringUtils.isNotEmpty((CharSequence)connSettings.databaseserver.userpassword)});
        config.setMaximumPoolSize(10);
        config.setMinimumIdle(2);
        config.setPoolName("HikariPool-" + connectionCode);
        config.addDataSourceProperty("cachePrepStmts", (Object)"true");
        config.addDataSourceProperty("prepStmtCacheSize", (Object)"250");
        config.addDataSourceProperty("prepStmtCacheSqlLimit", (Object)"2048");
        log.trace("Hikari pool properties set for code: {}", (Object)connectionCode);
        HikariDataSource dataSource = null;
        try {
            log.trace("Creating HikariDataSource instance for code: {}", (Object)connectionCode);
            dataSource = new HikariDataSource(config);
            log.debug("HikariDataSource instance created successfully for code: {}", (Object)connectionCode);
        }
        catch (Exception e) {
            log.error("Failed to create HikariDataSource for code '{}': {}", new Object[]{connectionCode, e.getMessage(), e});
            throw new Exception("Failed to create database connection pool for code: " + connectionCode, e);
        }
        this.dataSourcePools.put(connectionCode, dataSource);
        log.trace("Exiting getDataSource (created) for code: {}", (Object)connectionCode);
        return dataSource;
    }

    public Query createQuery(String connectionCode, String sql) throws Exception {
        Jdbi jdbi = this.getJdbi(connectionCode);
        return jdbi.open().createQuery(sql);
    }

    public synchronized Jdbi getJdbi(String connectionCode) throws Exception {
        log.trace("Entering getJdbi for code: {}", (Object)connectionCode);
        if (StringUtils.isBlank((CharSequence)connectionCode)) {
            log.error("getJdbi called with blank connection code.");
            throw new IllegalArgumentException("Connection code cannot be blank.");
        }
        if (this.jdbiInstances.containsKey(connectionCode)) {
            log.debug("Cache hit for Jdbi instance: {}", (Object)connectionCode);
            log.trace("Exiting getJdbi (cached) for code: {}", (Object)connectionCode);
            return this.jdbiInstances.get(connectionCode);
        }
        log.debug("Cache miss for Jdbi instance: {}. Creating new instance.", (Object)connectionCode);
        DataSource dataSource = this.getDataSource(connectionCode);
        if (dataSource == null) {
            log.error("getDataSource returned null for code: {}. Cannot create Jdbi instance.", (Object)connectionCode);
            throw new Exception("Failed to obtain DataSource for code: " + connectionCode);
        }
        log.trace("Creating Jdbi instance using DataSource for code: {}", (Object)connectionCode);
        Jdbi jdbi = Jdbi.create((DataSource)dataSource);
        log.debug("Jdbi instance created successfully for code: {}", (Object)connectionCode);
        this.jdbiInstances.put(connectionCode, jdbi);
        log.trace("Exiting getJdbi (created) for code: {}", (Object)connectionCode);
        return jdbi;
    }

    public synchronized void shutdownConnections() {
        log.info("Shutting down all managed database connection pools...");
        this.dataSourcePools.forEach((code, ds) -> {
            try {
                if (ds != null && !ds.isClosed()) {
                    log.debug("Closing DataSource pool for code: {}", code);
                    ds.close();
                } else {
                    log.trace("DataSource pool for code '{}' was already null or closed.", code);
                }
            }
            catch (Exception e) {
                log.error("Error closing DataSource pool for code '{}': {}", new Object[]{code, e.getMessage(), e});
            }
        });
        this.dataSourcePools.clear();
        this.jdbiInstances.clear();
        this.loadedConnectionDetails.clear();
        log.info("All connection pools shut down and caches cleared.");
    }

    public ServerDatabaseSettings getServerDatabaseSettings(String connectionCode) throws Exception {
        ConnectionDatabaseSettings settings = this.getConnectionSettings(connectionCode);
        if (settings == null || settings.databaseserver == null) {
            throw new IllegalStateException("No database-server settings found for code: " + connectionCode);
        }
        return settings.databaseserver;
    }
}

